# Driver instructions for the run test step of the continuous integration
#
# 1. Define parameters
# CAMITK_SITE="[Gitlab Runner] debian stable"                  # the name of the current machine 
# CAMITK_CI_MODE="Experimental"                                # ctest mode (Nightly, Continuous, Experimental...)
# CAMITK_CI_BRANCH="branch-name"                               # git branch name (check directly with git if not defined)
# CAMITK_CI_ID="Pipeline #$CI_PIPELINE_ID Job #$CI_BUILD_ID"   # unique id 
# CAMITK_SOURCE_DIR=~/Dev/CamiTK/src/camitk                    # path to CamiTK code source directory
# CAMITK_BUILD_DIR=~/Dev/CamiTK/build/camitk-exp               # path to the intended build directory
# CAMITK_BUILD_SETTINGS="GCC-64bits-Debug"                     # compiler-arch-buildtype string
#
# 2. run the command
# ctest -VV \
#       -DCTEST_SITE="$CAMITK_SITE" \
#       -DCI_MODE="$CAMITK_CI_MODE" \
#       -DCI_ID="$CAMITK_CI_ID" \
#       -DCI_BRANCH=$CAMITK_CI_BRANCH \
#       -DCI_BUILD_SETTINGS="$CAMITK_BUILD_SETTINGS" \
#       -DCTEST_SOURCE_DIRECTORY="$CAMITK_SOURCE_DIR" \
#       -DCTEST_BINARY_DIRECTORY="$CAMITK_BUILD_DIR" \
#       -DJUNIT_OUTPUT_DIRECTORY=some-dir-relative-to-build \
#       -S $CAMITK_SOURCE_DIR/sdk/cmake/ctest/ci-test.cmake > test.log 2>&1
#
# It will submit a new report in the "configure" section of the dashboard 
# identified as $SITE and $CAMITK_BUILD_SETTINGS.
#
# What this script does ?
# 1. INFORMATION STEP
#       Configure SITE and BUILD information to be correctly display on the dashboard
#       Loads information from the CTestConfig.cmake file.
# 2. RUN TEST
#       Run test for each CAMITK_TARGET of CamiTK

if (NOT CI_STAGE)
    set(CI_STAGE "Test")
endif()

# ------------------------ STEP 1: information step ------------------------
include("${CTEST_SOURCE_DIRECTORY}/sdk/cmake/ctest/ci-setup.cmake")
include("${CTEST_SOURCE_DIRECTORY}/sdk/cmake/ctest/ci-log.cmake")

# write log to a file
ci_start_log(FILENAME "${CI_PROJECT_LOG_DIRECTORY}/ci-test.log")

# ------------------------ STEP 2: test ------------------------
ci_log("Step 2. Test all CAMITK_TARGETs... Using ${NUMBER_OF_PROC} tests in parallel")

# set extra time for test to 30 min = 1800 s (default is 10 min = 600 s)
set(CTEST_TEST_TIMEOUT 1800)

# Get the CAMITK_TARGETs listing (automatically created at configuration step)
include("${CTEST_BINARY_DIRECTORY}/Subprojects.cmake")

# Update CDash configuration to the server
# The Project.xml file is automatically generated at configure time. 
# If a new extension is added to CamiTK, CDash will automatically be updated according to it.
# To do this, send this file to the server
ctest_submit(FILES "${CTEST_BINARY_DIRECTORY}/Project.xml") 

# Build each CAMITK_TARGETs
foreach(CAMITK_TARGET ${CAMITK_TARGETS})
        ci_log("- Setting tags for ${CAMITK_TARGET}...")

        # tag sub project
        set_property(GLOBAL PROPERTY SubProject ${CAMITK_TARGET})
        set_property(GLOBAL PROPERTY Label ${CAMITK_TARGET})

        # run tests (in parallel)
        ci_log("- Running tests for ${CAMITK_TARGET}...")

        if(${CMAKE_VERSION} VERSION_LESS "3.21.0")
            # OUTPUT_JUNIT was introduced in CMake 3.21.0
            ctest_test(BUILD "${CTEST_BINARY_DIRECTORY}" 
                       INCLUDE_LABEL "${CAMITK_TARGET}"
                       PARALLEL_LEVEL ${NUMBER_OF_PROC}    # Run test in parallel
                       TEST_LOAD ${NUMBER_OF_PROC}         # But not if load threshold is over the number of proc
                       RETURN_VALUE TEST_RES)
        else()
            ctest_test(BUILD "${CTEST_BINARY_DIRECTORY}" 
                       INCLUDE_LABEL "${CAMITK_TARGET}"
                       PARALLEL_LEVEL ${NUMBER_OF_PROC}    # Run test in parallel
                       TEST_LOAD ${NUMBER_OF_PROC}         # But not if load threshold is over the number of proc
                       OUTPUT_JUNIT ${JUNIT_OUTPUT_DIRECTORY}/${CAMITK_TARGET}.xml
                       RETURN_VALUE TEST_RES)
        endif()
        
        if (NOT TEST_RES EQUAL 0)
            ci_log(CI_ERROR "- Test for ${CAMITK_TARGET} failed with return value ${TEST_RES}")
        else()
            ci_log("- Test for ${CAMITK_TARGET} passed")
        endif()
        
        ci_log("- Submitting tests for ${CAMITK_TARGET}...")

        ctest_submit(PARTS Test RETURN_VALUE TEST_RES)
        
        if (NOT TEST_RES EQUAL 0)
            ci_log(CI_ERROR "- Submitting test results for ${CAMITK_TARGET} failed with return value ${TEST_RES}")
        else()
            ci_log("- Submitting test results for ${CAMITK_TARGET} passed")
        endif()
endforeach()

ci_log("Tests done")
