/*
 * // Copyright (c) Radzivon Bartoshyk 7/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
J0 zeros and extremums.

Generated by SageMath:
```python
# searching for zeros and extremums
R120 = RealField(120)

zeros = []

mp.prec = 150

step = mpf("0.001")
epsilon = mpf("1e-35")
x = mpf("0.0")

def j0_prime(x):
    return diff(lambda t: besselj(0, t), x)

previous_zero = R120(0)
j0_zeros = []

while x < mpf("76.0"):
    f1 = besselj(0, x)
    f2 = besselj(0, x + step)
    if f1 * f2 < 0:
        zero = findroot(lambda t: j0(t), (x, x + step), solver='bisect', tol=mp.mpf("1e-41"))
        previous_zero = zero
        j0_zeros.append(zero)
    if previous_zero is not None and abs(x - mpf(f'{round(x)}')) < epsilon:
        zeros.append(previous_zero)
    x += step

j0_extrema = []

x = mpf("0.0")
while x < mpf("76.0"):
    d1 = mp.diff(lambda t: j0(t), x)
    d2 = mp.diff(lambda t: j0(t), x + step)
    if d1 * d2 < 0:
        extremum = findroot(lambda t: mp.diff(lambda u: j0(u), t), (x, x + step), solver='bisect', tol=mp.mpf("1e-41"))
        j0_extrema.append(extremum)
    x += step

# Print results
for i, z in enumerate(j0_zeros):
    print(f"Zero {i+1}: x ≈ {z}")

print("Extrema (peaks/valleys) of J0(x):")
for e in j0_extrema:
    print(f"nExtrema: {e}")

j0_zeros.extend(j0_extrema)

j0_zeros = sorted(j0_zeros)

# Print results
for i, z in enumerate(j0_zeros):
    print(f"Peak or zero {i+1}: x ≈ {z}")

print("")

print("pub(crate) static J0_ZEROS: [(u64, u64); 48] = [")
print(f"(0x0, 0x0),")
for z in j0_zeros:
    k = split_double_double(z)
    hi = double_to_hex(k[1])
    lo = double_to_hex(k[0])
    print(f"({lo}, {hi}),")

print("];")
```
**/
pub(crate) static J0_ZEROS: [(u64, u64); 48] = [
    (0x0, 0x0),
    (0xbca0f539d7da258e, 0x40033d152e971b40),
    (0xbca60155a9d1b256, 0x400ea75575af6f09),
    (0x3c975054cd60a517, 0x4016148f5b2c2e45),
    (0xbc9b226d9d243827, 0x401c0ff5f3b47250),
    (0xbcb51970714c7c25, 0x40214eb56cccdeca),
    (0x3cc02610a51562b6, 0x402458d0d0bdfc29),
    (0x3cb444fd5821d5b1, 0x40279544008272b6),
    (0x3cb2bce7fd18e693, 0x402aa5baf310e5a2),
    (0xbcc9796609364e85, 0x402ddca13ef271d2),
    (0xbcdd2a68e88ab317, 0x4030787b360508c5),
    (0xbcd165fd108f46ff, 0x403212313f8a19f6),
    (0xbcd21830197e9e86, 0x40339da8e7416ca4),
    (0x3cc1d2dfa1c3b5a8, 0x4035362dd173f792),
    (0xbcc1bf33afef88f1, 0x4036c294e3d4d8ac),
    (0x3cd0847c620015e0, 0x40385a3b930156dd),
    (0x3cc1a2686480d882, 0x4039e7570dcea106),
    (0x3cdd2b3714972b28, 0x403b7e54a5fd5f11),
    (0xbcb42ce39ec976fb, 0x403d0bfcf471fccc),
    (0xbcc36bbabc1c9f31, 0x403ea27591cbbed2),
    (0xbcdbe3a1cd066b66, 0x404018476e6b2bf0),
    (0x3cda326cf4307839, 0x4040e34e13a66fe6),
    (0xbced5fbbff045068, 0x4041aa890dc5e97c),
    (0xbcd0b6068f861c6f, 0x404275637a9619ec),
    (0x3cc9eafeca0ca4fd, 0x40433cc523d5cb69),
    (0xbcb34c86f4e27936, 0x4044077a7ed6293a),
    (0x3cc489bd556e510a, 0x4044cefcf1734b62),
    (0x3cced48fe99f45ef, 0x40459992c65d0d8d),
    (0x3ce4f716f3179d90, 0x404661315d6b133f),
    (0xbcd05a7a0525058f, 0x40472bac0f810810),
    (0xbcef3950a842db79, 0x4047f36312028ad6),
    (0x3ce575dc7f8a031a, 0x4048bdc6293f0657),
    (0x3ce85d7bdb30baf1, 0x404985928f96d51e),
    (0xbcdfa16a338bbaee, 0x404a4fe0ee444c7b),
    (0x3ce3d41e041caa68, 0x404b17c038c2018c),
    (0xbce43e4a90356acf, 0x404be1fc41a4c607),
    (0x3cda139ce2cd08ac, 0x404ca9ec5a82324b),
    (0x3ce0f4b1c9544480, 0x404d74180c9e41f6),
    (0xbcb12e6ef2e594e1, 0x404e3c1731d64f1e),
    (0x3ceff10a69607aab, 0x404f06343d0971d4),
    (0x3cdfd1ee8286358a, 0x404fce40efb1156e),
    (0xbcf8991ca07c84c0, 0x40504c28621f11e6),
    (0x3ced3cacfc720419, 0x4050b034dde75b42),
    (0xbcee90a52cffc26f, 0x40511536cb22d72b),
    (0xbcee669304bfe748, 0x40517948db63675c),
    (0xbcf5328276c045f2, 0x4051de4554a1c2dd),
    (0x3cf8eb4a94b63936, 0x4052425c7dcacdf6),
    (0x3cff05f585843675, 0x4052a753fa820480),
];

/**
Roots and values for extremums and J0 zeros.

Generated by MPFR:
```text
let mut arr = vec![];
for zeros in J0_ZEROS.iter() {
    let mpfr = Float::with_val(107, f64::from_bits(zeros.1)).j0();
    arr.push(mpfr.to_f64().to_bits());
}
println!(
    "arr: [{}]",
    arr.iter()
        .map(|x| format!("0x{:016x}", x))
        .collect::<Vec<_>>()
        .join(", ")
);
```
**/
pub(crate) static J0_ZEROS_VALUE: [u64; 48] = [
    0x3ff0000000000000,
    0xbc919b7921f03c8e,
    0xbfd9c6cf582cbf7f,
    0xbc7fbb40985f6e34,
    0x3fd33518b3874e8a,
    0xbc96e8eeb22e5818,
    0xbfcff654544ebcd1,
    0xbc92d8ed368e0843,
    0x3fcbf3337873a7d8,
    0xbca50be2ef09843e,
    0xbfc925c6fca08f55,
    0x3caa2122af76659e,
    0x3fc70c511227d5aa,
    0x3c98b4b912da2218,
    0xbfc5664e13b70622,
    0xbca55e059345b430,
    0x3fc40f90793605bb,
    0x3cb1c17abe35eaae,
    0xbfc2f2072e638cf4,
    0x3c966608ac164dbd,
    0x3fc1ff5eec6a01cd,
    0x3cacc62c8409daa7,
    0xbfc12dd57bf18ada,
    0x3ca18e7b1f5c77ae,
    0x3fc076826cc2c191,
    0xbc8376cc57901722,
    0xbfbfa8b41711c83a,
    0xbc9df12b66b26573,
    0x3fbe8727daa3daed,
    0xbc9eab3b2bf5813d,
    0xbfbd8293aa55d18f,
    0xbcb3795064667200,
    0x3fbc96700bf039e2,
    0xbcabd526bce38ac7,
    0xbfbbbf246914235f,
    0x3cb14dbc2a692434,
    0x3fbaf9cb49c4f935,
    0x3cac341b9f831174,
    0xbfba4407e04298d1,
    0xbcb9e23f9c9ca91e,
    0x3fb99be744018c90,
    0xbcc3725f95922088,
    0xbfb8ffc9bd24fe08,
    0x3cb79a2187735244,
    0x3fb86e51be0a9153,
    0xbcc0012df1d3ef73,
    0xbfb7e656efb009ae,
    0xbcc6ecd27843db59,
];

/**
Taylor coefficients for J0 [0; 74.62]

Generated by SageMath with Sollya:
```python
def compute_intervals(zeros):
    intervals = []
    for i in range(0, len(zeros)):
        if i == 0:
            a = (zeros[i]) / 2 - 0.05 - zeros[i]
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.05 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        elif i + 1 > len(zeros) - 1:
            a = (zeros[i - 1] + zeros[i]) / 2 - 0.05 - zeros[i]
            b = (zeros[i]) + 0.83 + 0.05 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        else:
            a = (zeros[i - 1] + zeros[i]) / 2 - zeros[i] - 0.05
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.05  - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
    return intervals

intervals = compute_intervals(j0_zeros)
# print(intervals)

def build_sollya_script(a, b, zero, deg):
    return f"""
prec = 500;
bessel_j0 = library("./notes/bessel_sollya/cmake-build-release/libbessel_sollya.dylib");
f = bessel_j0(x + {zero});
d = [{a}, {b}];
pf = remez(f, {deg}, d);
for i from 0 to degree(pf) do {{
    write(coeff(pf, i)) >> "coefficients.txt";
    write("\\n") >> "coefficients.txt";
}};
"""

def load_coefficients(filename):
    with open(filename, "r") as f:
        return [RR(line.strip()) for line in f if line.strip()]

def call_sollya_on_interval(a, b, zero, degree=12):
    sollya_script = build_sollya_script(a, b, zero, degree)
    with open("tmp_interval.sollya", "w") as f:
        f.write(sollya_script)
    import subprocess
    if os.path.exists("coefficients.txt"):
        os.remove("coefficients.txt")
    try:
        result = subprocess.run(
            ["sollya", "tmp_interval.sollya"],
            check=True,
            capture_output=True,
            text=True
        )
    except subprocess.CalledProcessError as e:
        return

degree = 13

print(f"pub(crate) static J0F_COEFFS: [[u64;{degree + 1}]; {len(intervals)}] = [")
for i in range(0, len(intervals)):
    interval = intervals[i]
    call_sollya_on_interval(interval[0], interval[1], interval[2], degree)
    coeffs = load_coefficients(f"coefficients.txt")
    print("[")
    for c in coeffs:
        print(double_to_hex(c) + ",")
    print("],")
print("];")
```
**/
pub(crate) static J0F_COEFFS: [[u64; 14]; 47] = [
    [
        0xbca69de51bcc0120,
        0xbfe09cdb3655127d,
        0x3fbba1deea029925,
        0x3facfae8643687e7,
        0xbf81bb1cbe1caeb6,
        0xbf61f992590d3f89,
        0x3f315382bbf1a06b,
        0x3f06ed3ba591e6da,
        0xbed232c8d8aba5cc,
        0xbea1cceb52db45ee,
        0x3e68006b6b5682ca,
        0x3e329cc1da0793f0,
        0xbdf5dbb0053da32b,
        0xbdc04e14e5d96539,
    ],
    [
        0xbfd9c6cf582cbf7f,
        0x3cb1d608597b5d8f,
        0x3fc9c6cf582cbf55,
        0xbf91f06d14e12b29,
        0xbf8b589d1da0f7e4,
        0x3f50f9103d00ae93,
        0x3f38644561ce9290,
        0xbefa2a034c749192,
        0xbed83a0686fd1dd2,
        0x3e96a5085d7844b5,
        0x3e6ebfcaacd49b87,
        0xbe2964b9932b57f3,
        0xbdfad73cf4701511,
        0x3db5acbea609d3e5,
    ],
    [
        0xbc6a1105f9dbcdd5,
        0x3fd5c6e60a097826,
        0xbf9f8f72e7a8471e,
        0xbfab2150cb41ece4,
        0x3f72f7ffe901b59c,
        0x3f627e31fe9ddafb,
        0xbf26f641f369bbbd,
        0xbf0863f485fc7e89,
        0x3ecad77cbbb32bcd,
        0x3ea32e705af41964,
        0xbe62d9d2bb447e35,
        0xbe341f0e4f4dc44a,
        0x3df198b51eee9749,
        0x3dbec8f104a030ca,
    ],
    [
        0x3fd33518b3874e8a,
        0xbca7f56ee546a569,
        0xbfc33518b3874e3b,
        0x3f7d34125d59fb96,
        0x3f880c83bdee71b4,
        0xbf4483c20f2abbd1,
        0xbf36ffa5fc0ccc0f,
        0x3ef2ccf7bbc1d620,
        0x3ed796a6caa245fc,
        0xbe91e85738cfdd42,
        0xbe6e69b581df4be3,
        0x3e2550287122730e,
        0x3dfab73cc2e87ac7,
        0xbdb250809ed630ba,
    ],
    [
        0x3c684670459725e1,
        0xbfd15f7977a772d5,
        0x3f900f7fcf183c65,
        0x3fa68b984ec64b1d,
        0xbf648e63600c54a1,
        0xbf60e0d6038721a4,
        0x3f1d7960512fb682,
        0x3f07800bc7410c66,
        0xbec3324799a7e0fb,
        0xbea30e8df2325ca1,
        0x3e5cecefe3d75abb,
        0x3e3458cbaa06ef8d,
        0xbdec4dd6cf551d07,
        0xbdbefa0975b65713,
    ],
    [
        0xbfcff654544ebcd1,
        0x3ca0ffc636e7373a,
        0x3fbff654544ebc20,
        0xbf70c17ff72b1352,
        0xbf84b0c5d5d9e7c9,
        0x3f394154be849b76,
        0x3f34e12c2fdccc3a,
        0xbee9f32fcfbab067,
        0xbed63c53e4cc158b,
        0x3e8adbb88b7826d7,
        0x3e6d5f825ac22ee0,
        0xbe20f17a50351084,
        0xbdfa326458f60fce,
        0x3dae0e0c6fa040ee,
    ],
    [
        0xbc648b5c145d146f,
        0x3fcdc13e66ac2e78,
        0xbf842ff0cdc58194,
        0xbfa38d1dd8992ef5,
        0x3f5a55e9b344ea02,
        0x3f5e2e16f97e7f5f,
        0xbf13dfc37714f92e,
        0xbf05ce7f4a51d917,
        0x3ebbb177bc7895b4,
        0x3ea2346df84ab988,
        0xbe56125b44798bd5,
        0xbe33d7cc75fbdd4e,
        0x3de68bc84d67bb0a,
        0x3dbe749161c75b7f,
    ],
    [
        0x3fcbf3337873a7d8,
        0xbc999eb31028b75f,
        0xbfbbf3337873a725,
        0x3f66604d91f94ba7,
        0x3f8251858010ffd8,
        0xbf314bc11a41832e,
        0xbf32e7decc933381,
        0x3ee293b4d39e1d9e,
        0x3ed4a66fbc1f8e7c,
        0xbe843cb02661be36,
        0xbe6bdd61e41af9d3,
        0x3e1ab24db7abbdd7,
        0x3df940cc3ecafe33,
        0xbda870435d8d1863,
    ],
    [
        0x3c6110461d393f0c,
        0xbfca701d0f967501,
        0x3f7c54b930fef3e4,
        0x3fa17798aa09f194,
        0xbf52a21514062f70,
        0xbf5b541f829ca99f,
        0x3f0cc0bd9fd11443,
        0x3f041f3b06c79efa,
        0xbeb4b22fc3292c98,
        0xbea1223e5773d27c,
        0x3e5118e5cf14a591,
        0x3e32ffb6478c2281,
        0xbde20a2b34bd7e9f,
        0xbdbd77afc051bc6c,
    ],
    [
        0xbfc925c6fca08f54,
        0x3c941bba0a9908a3,
        0x3fb925c6fca08ea3,
        0xbf6049377403d97d,
        0xbf809463bbcfb70d,
        0x3f297b35471ddd4a,
        0x3f314dd43fc4bda8,
        0xbedbf665e54737fe,
        0xbed32caf8123dd12,
        0x3e7f510484fa37ba,
        0x3e6a42a3b8b1effd,
        0xbe1544b8d56d77a8,
        0xbdf81cfb72892ddb,
        0x3da3f5749b01c5c3,
    ],
    [
        0xbc5c64a3611bf4cb,
        0x3fc8077f56c9b782,
        0xbf75467eb535dac4,
        0xbf9fd7c3ad6f5a3e,
        0x3f4c1b47c806f9c6,
        0x3f59166c7d3ee8f7,
        0xbf05f01522f72552,
        0xbf02aa939ff2e34a,
        0x3eb0129642c0eedc,
        0x3ea01716d7274ce0,
        0xbe4b21ada69ec1ec,
        0xbe320ffbd6f367e8,
        0x3ddd441699eaa4df,
        0x3dbc46351a30e033,
    ],
    [
        0x3fc70c511227d5aa,
        0xbc903dd711e6b7a7,
        0xbfb70c511227d4fe,
        0x3f5910ebe1f1fa45,
        0x3f7e7dc08e6ff244,
        0xbf23bd7d15b0a88f,
        0xbf300357a101d595,
        0x3ed5ec733cb20d5b,
        0x3ed1e5d1fa4386ae,
        0xbe78f423a07fedeb,
        0xbe68c089aeb3a360,
        0x3e1143a2a87b28be,
        0x3df6f2d06548607b,
        0xbda07d7d6f49ddb9,
    ],
    [
        0x3c57d1baaf487c60,
        0xbfc62d93aa9d05bb,
        0x3f70ba9ce88926ac,
        0x3f9d7073daebb038,
        0xbf462813c7f32d76,
        0xbf574a948d055ffc,
        0x3f01695764ad01fd,
        0x3f0170ab5ed77aa3,
        0xbea9c8e2cba34328,
        0xbe9e4d88f9ebcadb,
        0x3e460ec3e9bbfca2,
        0x3e312751ad8cf2c9,
        0xbdd825b2017389c1,
        0xbdbb0b4f2fdaca95,
    ],
    [
        0xbfc5664e13b70621,
        0x3c8adfee580612ab,
        0x3fb5664e13b7057b,
        0xbf540ee3940b2f9d,
        0xbf7c5e1ad9fa40ad,
        0x3f1fb8a98f136a0c,
        0x3f2de9be57a255f8,
        0xbed1bec95e73b04b,
        0xbed0cf25bf302fd8,
        0x3e746784408bf531,
        0x3e676640f349bd8a,
        0xbe0c944164ad511b,
        0xbdf5d837e7a07aba,
        0x3d9ba3ab0edd6dab,
    ],
    [
        0xbc543722dc20f623,
        0x3fc4b2a2ebf61ecd,
        0xbf6b3297fdae7373,
        0xbf9b8105d59b1125,
        0x3f420a3f8c10a2c5,
        0x3f55d18d69de4109,
        0xbefc79db490d157b,
        0xbf00679c92947f5d,
        0x3ea53ac3d1de0eb7,
        0x3e9ca750128116cf,
        0xbe4252ac12ff61fa,
        0xbe3051067c8a6420,
        0x3dd4435be57e146d,
        0x3db9dcd0ca8a2a1d,
    ],
    [
        0x3fc40f90793605bb,
        0xbc86a24c27e99837,
        0xbfb40f907936051b,
        0x3f5085775a554a25,
        0x3f7aa0ce0420ebfd,
        0xbf1a32a28e65afde,
        0xbf2c26ebc916e22b,
        0x3ecd740098d21c07,
        0x3ecfc1baf78438dd,
        0xbe710c76235f5aa1,
        0xbe663455427d9c57,
        0x3e08141f4d843200,
        0x3df4d5216dec0df1,
        0xbd977e29b16d5fdb,
    ],
    [
        0x3c5158813dc387f0,
        0xbfc37aac8c1aeabb,
        0x3f66ac0d2e2f2ac1,
        0x3f99e74e754ea6d8,
        0xbf3e1c0589dfbd21,
        0xbf5496158dc5b2be,
        0x3ef7d554035ae72e,
        0x3eff0b30f3d1fb7e,
        0xbea1d9e26f5a608f,
        0xbe9b35eeaebb5fed,
        0x3e3f010034c79a0b,
        0x3e2f1f31b65b3b40,
        0xbdd1445486f07cd4,
        0xbdb8c39615e71112,
    ],
    [
        0xbfc2f2072e638cf3,
        0x3c8361a1f631cd41,
        0x3fb2f2072e638c5a,
        0xbf4bd42b64fc9353,
        0xbf792bb5e1e07ce1,
        0x3f161ace339d0b73,
        0x3f2aa8d1ce9fa384,
        0xbec8ef625b21b762,
        0xbece26d28032c92c,
        0x3e6d0116b3ae24e4,
        0x3e6526a75ee1fb24,
        0xbe049a7cd1a8b231,
        0xbdf3eac7c4ed94be,
        0x3d943b63ca10efa6,
    ],
    [
        0xbc4e15276b2257f6,
        0x3fc27407dfadee6d,
        0xbf6346950bfd8dcc,
        0xbf988d48d1d4eb20,
        0x3f399e6923aaadf2,
        0x3f538984b76c8a40,
        0xbef4521949da5854,
        0xbefd855d7afc24f7,
        0x3e9e8a86b94090aa,
        0x3e99f1bce3767a28,
        0xbe3aa3a4df470ca3,
        0xbe2dc43a0da754d3,
        0x3dcdd3f81803822b,
        0x3db7c1fb1e948790,
    ],
    [
        0x3fc1ff5eec6a01cd,
        0xbc80d20b3aa154f7,
        0xbfb1ff5eec6a0139,
        0x3f47daf64983dfb0,
        0x3f77ed5fffc0f26a,
        0xbf12f9479636506c,
        0xbf296027e938174a,
        0x3ec57486d36f86d5,
        0x3eccc1196ea00471,
        0xbe690ae6942f1564,
        0xbe64382bea1931c6,
        0x3e01ddd9b1b7300f,
        0x3df317ad2d0ae68b,
        0xbd91a1072249bec4,
    ],
    [
        0x3c4a5ad4fde0c9d8,
        0xbfc192f23ce3e050,
        0x3f60a668185bfe0f,
        0x3f9764141d652022,
        0xbf3624437a2d49f0,
        0xbf52a184be0d3982,
        0x3ef196de0d85b64c,
        0x3efc317f84a19048,
        0xbe9a80187c3c9493,
        0xbe98d3840a83aa5f,
        0x3e3730c02a4a94ac,
        0x3e2c8d1491c62663,
        0xbdca1189016a9692,
        0xbdb6d754fe28a395,
    ],
    [
        0xbfc12dd57bf18ad9,
        0x3c7d8e82da6b051b,
        0x3fb12dd57bf18a4a,
        0xbf44bebeff7ba93d,
        0xbf76d9afe882346d,
        0x3f10842d50795e09,
        0x3f2841d86abe55fe,
        0xbec2b5cab87b5693,
        0xbecb86b9f14c3df8,
        0x3e65e518bb040851,
        0x3e63642ad66ee13c,
        0xbdff581d7497507c,
        0xbdf2596d87b9a5f8,
        0x3d8f0ac4f8bd7787,
    ],
    [
        0xbc47483488f843f0,
        0x3fc0cf3ed059c573,
        0xbf5d242aa5298caf,
        0xbf96613d93b0179b,
        0x3f33627f261d008d,
        0x3f51d69ca0d81e39,
        0xbeeed574ad2bb0d3,
        0xbefb06384da1c6bc,
        0x3e97452b1c4c76e9,
        0x3e97d51e823013c9,
        0xbe346a18e9324483,
        0xbe2b754cf5bbe0eb,
        0x3dc70467e00ca1a0,
        0x3db601d42e3f412d,
    ],
    [
        0x3fc076826cc2c191,
        0xbc7a319136b9885b,
        0xbfb076826cc2c107,
        0x3f4241b03eab1b46,
        0x3f75e7f530011e0e,
        0xbf0d17978e4af2b0,
        0xbf2745b0deaaa29c,
        0x3ec0803f899002ed,
        0x3eca70060b7e6d91,
        0xbe635913c3c95352,
        0xbe62a694daaaf841,
        0x3dfbc4a0d8050004,
        0x3df1ad85820a38f2,
        0xbd8b942dd09eb308,
    ],
    [
        0x3c44bc1f080e9694,
        0xbfc0230b9797a7b2,
        0x3f59c8083b2b6f7f,
        0x3f957d3203befd1b,
        0xbf3127cba226824c,
        0xbf5123447144f140,
        0x3eeb4fe26c88e56e,
        0x3ef9fc5254458847,
        0xbe94a44b3f692077,
        0xbe96f17d3ebae645,
        0x3e3224c9056773e9,
        0x3e2a78ba4580cd4b,
        0xbdc481589f7c80a5,
        0xbdb53f51ba6e502a,
    ],
    [
        0xbfbfa8b41711c839,
        0x3c776c1e8f7fbdf3,
        0x3fafa8b41711c72d,
        0xbf403a8d0f113157,
        0xbf7511c6dad9e9ce,
        0x3f09e040fc7d8bd2,
        0x3f266582f59eaade,
        0xbebd62a1a025f4b0,
        0xbec976fa2e25f4ab,
        0x3e61411e4c00a8c3,
        0x3e61fc07a47b587f,
        0xbdf8d00a9e1b01e2,
        0xbdf1119f53145e82,
        0x3d88b3165222eaa1,
    ],
    [
        0xbc429a344bf53c30,
        0x3fbf13faf32c8e0a,
        0xbf570558dddb7821,
        0xbf94b24d7a9338fa,
        0x3f2ea52a2144c44d,
        0x3f50834d8f3f71fa,
        0xbee86941a2b2f276,
        0xbef90e32ce46341a,
        0x3e92785e5b786d2a,
        0x3e9624822c6837c1,
        0xbe304286df6c8e25,
        0xbe2993b44677fad8,
        0x3dc2692b9cb4de6f,
        0x3db48dab755272e9,
    ],
    [
        0x3fbe8727daa3daec,
        0xbc75105fe04e17a2,
        0xbfae8727daa3d9e9,
        0x3f3d19c52e0749cd,
        0x3f74524d481311c0,
        0xbf0735f790cd464b,
        0xbf259c8f9e41a823,
        0x3eba61a00c8699ad,
        0x3ec896d70eee04c2,
        0xbe5f04fa0ee2b2df,
        0xbe6161b9029aa2c1,
        0x3df656dc5e9fcd6d,
        0x3df083ab26947d7d,
        0xbd8646da85095b0a,
    ],
    [
        0x3c40c6e5f0cb085b,
        0xbfbe018d99f5da1b,
        0x3f54b85897b35dc0,
        0x3f93fc44215342e3,
        0xbf2b9694d7112c4c,
        0xbf4fe6fdc6440535,
        0x3ee5fd096c74d7eb,
        0x3ef83770c8f9fca0,
        0xbe90a6f608ea2259,
        0xbe956ad410ba57f4,
        0x3e2d5aedc8414c7c,
        0x3e28c319213f9f62,
        0xbdc0a501a42574f9,
        0xbdb3eaef334996a5,
    ],
    [
        0xbfbd8293aa55d18f,
        0x3c731f5b2ff0d80c,
        0x3fad8293aa55d093,
        0xbf3a48fe4aff1369,
        0xbf73a5ccbc11f183,
        0x3f04f91e4204c18c,
        0x3f24e72224187971,
        0xbeb7dac82ed8c06b,
        0xbec7cbd3d4a2512f,
        0x3e5c13a012cfa9d9,
        0x3e60d55d3c7a1e36,
        0xbdf43f07dbc0afd9,
        0xbdf001e1c5b3f199,
        0x3d8438754303f506,
    ],
    [
        0xbc3e721cd5615376,
        0x3fbd09b210b30217,
        0xbf52c74f6d11fe58,
        0xbf9357bfc2be57e6,
        0x3f2901e4c492a05d,
        0x3f4ee2a36979201d,
        0xbee3f0cb91fe387d,
        0xbef7748920d95467,
        0x3e8e399ed3001c0e,
        0x3e94c1b715414ff1,
        0xbe2aab03479a9f97,
        0xbe2804415514f135,
        0x3dbe479692aefaf1,
        0x3db3555d82fea6b2,
    ],
    [
        0x3fbc96700bf039e1,
        0xbc71712327ccc0e1,
        0xbfac96700bf038ec,
        0x3f37e5647d31307c,
        0x3f73095734a194a7,
        0xbf0312a4db676954,
        0xbf24424a9628de60,
        0x3eb5b4a67102dd95,
        0x3ec712e41250bc42,
        0xbe59918677637160,
        0xbe60550f5e9164bb,
        0x3df2750c65ef60c0,
        0x3def157ced5a6a13,
        0xbd8274ea409a4ed7,
    ],
    [
        0x3c3bc4a30293d67b,
        0xbfbc28612a3bc18a,
        0x3f511f52577ff2df,
        0x3f92c21da135f4f3,
        0xbf26ce18f81fd4cd,
        0xbf4df586d8b6aeb3,
        0x3ee230fedd1a2b84,
        0x3ef6c2a754e321b0,
        0xbe8b973110796ae8,
        0xbe9426ec6d82a4f4,
        0x3e285d2ec3ed03c3,
        0x3e2754eefd1cc277,
        0xbdbbb13bcfd1095a,
        0xbdb2cb6d2f5a71d0,
    ],
    [
        0xbfbbbf246914235e,
        0x3c6ff848b8004f4e,
        0x3fabbf2469142270,
        0xbf35d923e8472f22,
        0xbf727a96f1740b8b,
        0x3f01715e4bcd425f,
        0x3f23abacda98c7ab,
        0xbeb3dc30debc954f,
        0xbec6698cb2175b00,
        0x3e57691976ae1f76,
        0x3e5fbe7a90c59bea,
        0xbdf0ea1d3534d622,
        0xbdee39eb617c4576,
        0x3d80ee68d9ec5553,
    ],
    [
        0xbc396e9c97e8f53c,
        0x3fbb5a6219b35e14,
        0xbf4f645fdb1a8578,
        0xbf923940d01de870,
        0x3f24e86a1e60f65a,
        0x3f4d1c6a18c716ef,
        0xbee0aeec5fff60bb,
        0xbef61f7d2f793aca,
        0x3e8950f6140dd479,
        0x3e939898b4cca68e,
        0xbe265f0fa2e22f9c,
        0xbe26b33d2f34a7ca,
        0x3db9731a4bc752ba,
        0x3db24bcd6b2996ce,
    ],
    [
        0x3fbaf9cb49c4f934,
        0xbc6d6d35ce7e44b9,
        0xbfaaf9cb49c4f84c,
        0x3f3413b75ce0f622,
        0x3f71f7a8fec644b8,
        0xbf000844274ab940,
        0xbf23215daac1a429,
        0x3eb242e9f204ac2f,
        0x3ec5cdc3d7570c8d,
        0xbe5589d6656d4bd3,
        0xbe5ee52f2093b13f,
        0x3def25a4c863585f,
        0x3ded6edb90588bb5,
        0xbd7f34526b242536,
    ],
    [
        0x3c37650e44849b1d,
        0xbfba9d1835947d6f,
        0x3f4cea253049a1d9,
        0x3f91bb71f665dc68,
        0xbf23427f47955327,
        0xbf4c54a7bd6e5c2f,
        0x3edebe9e62f23b54,
        0x3ef58924f95f83e9,
        0xbe875643b068495f,
        0xbe93152fe242208c,
        0x3e24a270e811a32a,
        0x3e261d90f63d69a9,
        0xbdb77dbf375b3033,
        0xbdb1d55983193fd8,
    ],
    [
        0xbfba4407e04298d1,
        0x3c6b2fc36b7bca2c,
        0x3faa4407e04297ee,
        0xbf3288694b34d1f5,
        0xbf717f0266da7e64,
        0x3efd9a9a1d244e54,
        0x3f22a1c915f39e35,
        0xbeb0dd9e9cd3be0d,
        0xbec53dd8bf4c31dc,
        0x3e53e6ebf97ad18c,
        0x3e5e1c0952edd05e,
        0xbdeccc874da25bb3,
        0xbdecb26ccd2d18eb,
        0x3d7cdf4dbb3c6f9d,
    ],
    [
        0xbc359685a4f3dcbc,
        0x3fb9ee5ee937fc88,
        0xbf4abf28ad5bf0d6,
        0xbf9147481084ad98,
        0x3f21d137345ad0c1,
        0x3f4b9c10ddf55248,
        0xbedc72c9d23eafc6,
        0xbef4fe0b3595e36e,
        0x3e859a1afaceaec9,
        0x3e929b65a8c65855,
        0xbe231c3162eaeee6,
        0xbe25928cb280552e,
        0x3db5c4ff841acff7,
        0x3db167146fe195c5,
    ],
    [
        0x3fb99be744018c90,
        0xbc694d1606868f19,
        0xbfa99be744018bb2,
        0x3f312d4e1c1cc8e5,
        0x3f710f5ca51ef993,
        0xbefb714174938907,
        0xbf222b9fa834b14b,
        0x3eaf47107520863c,
        0x3ec4b861783e33d4,
        0xbe5276456457a7e2,
        0xbe5d612d705e69ee,
        0x3deabad16a1c7f4d,
        0x3dec0302ce313ca1,
        0xbd7ad13436da91d1,
    ],
    [
        0x3c33fe4be92d3d54,
        0xbfb94c6f54aef04b,
        0x3f48d6371f01895c,
        0x3f90db975fd7dbd2,
        0xbf208bd163414db7,
        0xbf4af0d3d4cc449e,
        0x3eda6c8e7a6a50d9,
        0x3ef47cddcd0777cf,
        0xbe8412382d315a8f,
        0xbe922a214336182c,
        0x3e21c37df131e495,
        0x3e25110597bc5108,
        0xbdb43f276f813535,
        0xbdb10022868bfde2,
    ],
    [
        0xbfb8ffc9bd24fe07,
        0x3c67889977473d30,
        0x3fa8ffc9bd24fd2f,
        0xbf2ff51b38ef3764,
        0xbf70a7a725d36032,
        0x3ef988128742f09f,
        0x3f21bdc845f948f5,
        0xbead1b60b68e1dce,
        0xbec43c2ce2f40a20,
        0x3e512fdc8d93240a,
        0x3e5cb30713086e1b,
        0xbde8e53a40557273,
        0xbdeb5f3c07e7903e,
        0x3d78fd3f37268235,
    ],
    [
        0xbc3292fab12602d5,
        0x3fb8b5ccad12d631,
        0xbf4724d018597ba3,
        0xbf907764ae2b1e77,
        0x3f1ed6acc18c1a89,
        0x3f4a51693df22314,
        0xbed8a0ec5fd8948c,
        0xbef4047f31589663,
        0x3e82b667a039f6a4,
        0x3e91c073e21e1306,
        0xbe209143960c8785,
        0xbe2497fb13bc2530,
        0x3db2e4699f6adc63,
        0x3db09fc4db2b9fa9,
    ],
    [
        0x3fb86e51be0a9153,
        0xbc65fa6f95d443a0,
        0xbfa86e51be0a907f,
        0x3f2dd3c244b5713f,
        0x3f7046fc5a20f248,
        0xbef7d51accd8d1f8,
        0xbf2157556b6a5aec,
        0x3eab2d01d207a4bc,
        0x3ec3c837e1b45aa2,
        0xbe500d3d097d314b,
        0xbe5c103c5b331300,
        0x3de742b129024e6b,
        0x3deac5e6c8a70206,
        0xbd775bdd14ccde28,
    ],
    [
        0x3c314cbdbd279267,
        0xbfb829356999a096,
        0x3f45a280e033e683,
        0x3f9019dba8336dd6,
        0xbf1cd4559d92616b,
        0xbf49bc85774f69b2,
        0x3ed70706561ca508,
        0x3ef393fc6c586d81,
        0xbe818009f70f6704,
        0xbe915d911e7ad3a2,
        0x3e1eff8fd0ef6949,
        0x3e24268fd4573ff9,
        0xbdb1ae6de683a236,
        0xbdb0455820923c5a,
    ],
    [
        0xbfb7e656efb009ad,
        0x3c649ee393402ee6,
        0x3fa7e656efb008de,
        0xbf2bec6b33f0062e,
        0xbf6fd932c269835b,
        0x3ef6504d7e1cd7db,
        0x3f20f77ce56ff037,
        0xbea972e583ac028d,
        0xbec35ba4e79f8507,
        0x3e4e1250b982c251,
        0x3e5b77a3c340d6b3,
        0xbde5cbcf474ee83d,
        0xbdea35fa30652af1,
        0x3d75e5f83583c368,
    ],
    [
        0xbc375f30b05ffab2,
        0x3fb7a597e9550932,
        0xbf44486c0b011f0f,
        0xbf8f848eec0e0c40,
        0x3f1b077fae02763b,
        0x3f49310d6e6682f5,
        0xbed597a5bccfeb73,
        0xbef32a855d113180,
        0x3e8069b8fa206f26,
        0x3e9100c89784eeac,
        0xbe1d14a175c009c5,
        0xbe23bbc9bff27ef1,
        0x3db0950fae2cbe6f,
        0x3dafc6492b94025c,
    ],
];
